/**********************************************************************
 * 
 *  Toby Opferman
 *
 *  Mutiny Driver!
 *
 *  This driver implements Operating System Mutiny!
 *  
 *  This example is for educational purposes only.  I license this source
 *  out for use in learning how to write a device driver.
 *
 *     Driver Entry Point  Copyright (c) 2005, All Rights Reserved
 **********************************************************************/




#ifndef __GDT_H__
#define __GDT_H__

    
struct _OPERATING_SYSTEM_THUNK_DATA;
typedef struct _OPERATING_SYSTEM_THUNK_DATA *POPERATING_SYSTEM_THUNK_DATA;

#define GDT_MAX_ENTRIES                  20

/**********************************************************
 * Descriptor Entry Constants
 *
 *   The following describe the contants used to set
 *   data on the bit entries in the GDT descriptor entries
 *
 *   Brief descriptions are included however for more information
 *   simply consult the INTEL and AMD processor reference manuals.
 **********************************************************/
 
 /*
  * Descriptor Privlege Level
  *    
  *    0 = Most Privledge, 3 = Least Privledge
  */
#define  GDT_DPL_0     0x00
#define  GDT_DPL_1     0x20
#define  GDT_DPL_2     0x40
#define  GDT_DPL_3     0x60
#define  GDT_DPL_MASK  0x9F

 /*
  * System Bit
  *
  *  This helps to define the type of descriptor.
  *    
  *    1 = User
  *        Types = Code, Data
  *
  *    0 = System
  *        Types = LDT, TSS, Call gate
  *     
  */
#define  GDT_SYSTEM_BIT      0x10
#define  GDT_SYSTEM_BIT_MASK 0xEF

/*
 * Present Bit
 *
 *   Informs the CPU that this segment is loaded into memory.
 *   If this bit is not set then an exception occurs.
 *
 */ 
 
#define  GDT_PRESENT_BIT       0x80
#define  GDT_PRESENT_BIT_MASK  0x7F
    
/* Granularity Bit
 *
 *   1 = The value specified by "Segment Limit" is in Pages
 *   0 = The value specified by "Segment Limit" is in Bytes.
 *
 * This helps to describe the granularity of "Segment Limit"
 * entry in the descriptor.  If you notice it isn't a 32 bit number
 * so if it was only allowed to specify bytes then you could not
 * describe a 4 Gigabyte segment.  Through enabling "Pages" you
 * can describe up to 4 Gigabyte segments.
 *
 */
#define  GDT_GRANULARITY_PAGE_BIT 0x80
#define  GDT_GRANULARITY_BYTE_BIT 0x00
#define  GDT_GRANULARITY_MASK     0x7F


/* Available To OS
 *
 *   This bit is available to the OS.
 *
 */
#define  GDT_AVAILABLE_TO_OS_BIT  0x10
#define  GDT_AVAILABLE_TO_OS_MASK 0xEF

/* Default Instruction Size
 *
 * Specifies the default instruction operand size such as
 * "B8" is "MOV AX", "MOV EAX" or "MOV RAX".
 *
 * In 32 Bit Protected Mode:
 *   1 = 32 bit
 *   0 = 16 bit
 *
 * In Long Mode:
 *   1 = 32 bit
 *   0 = 64 bit 
 *  
 */
#define  GDT_DEFAULT_INSTRUCTION_SIZE_32   0x40
#define  GDT_DEFAULT_INSTRUCTION_SIZE_16   0x00
#define  GDT_DEFAULT_INSTRUCTION_SIZE_MASK 0xBF

             
/*
 *  Conforming Bit
 *
 *  1 = Conforming
 *  0 = Not Conforming
 *
 *   This has to do with privledge level.  CPL = Current Privledge Level
 *   
 *   If a lower privledge segment branches to a higher privledge segment,
 *   if the higher privledge segment is "conforming" then the CPL does NOT
 *   change.
 *
 *   If a lower privledge segment attempts to branch to a NON-Conforming
 *   higher privledge segement, this cannot happen without the use of
 *   a Call-Gate.
 *
 *
 */
#define  GDT_CONFORMING_BIT  0x04
#define  GDT_CONFORMING_MASK 0xFB


/*
 *  Read/Execute Bit
 *
 *  1 = Read/Execute
 *  0 = Execute Only
 *
 *  This bit implements protection from reading a code segment.
 *
 *
 */
#define  GDT_READ_EXECUTE_BIT 0x02
#define  GDT_EXECUTE_ONLY_BIT 0x00
#define  GDT_READ_EXE_MASK    0xFD


/*
 *  Accessed Bit
 *
 *  1 = Accessed
 *  0 = Not Accessed
 *
 *  This bit is set by the OS.
 *
 *
 */
#define  GDT_ACCESSED_BIT  0x01
#define  GDT_ACCESSED_MASK 0xFE



/*
 *  Long Mode
 *
 *  1 = Long Mode
 *  0 = Compatibility Mode
 *
 *  This bit describes if the current code segment is operating in 
 *  Longmode or Compatibility mode.
 *
 *
 */
#define  GDT_LONG_MODE_ENABLE 0x20
#define  GDT_LONG_MODE_MASK   0xDF

/*
 *  Code/Data Descriptor
 *
 *  These settings are used on the flags to help specify
 *  whether the descriptor is code or data.
 *
 *
 */
#define  GDT_CODE_DESCRIPTOR            0x18
#define  GDT_DATA_DESCRIPTOR            0x10
#define  GDT_CODE_DATA_DESCRIPTOR_MASK  0xE7


/**********************************************************
 * GDT System Descriptor Entry
 *
 *    This is used for call gates, task descriptors and LDT. 
 *
 * 31              24  23  22  21   20  19    16  15 14 13  12  11   8  7         0
 * +-----------------+---+---+----+---+---------+---+-----+---+-------+-----------+
 * | Base Address   | G | DB | L | AVL| Limit  | P | DPL | S | Type  | Base Addr  |
 * +------------------------------------------+----------------------------------+
 * |        Base Address 15 - 0              |     Segment Limit 15 - 0         |
 * +----------------------------------------------------------------------------+
 *
 *  wSegmentLength                   = 0  - 15 (Segment Limit bits 15 - 0)
 *  wBaseAddressLowWord              = 16 - 31 (Base Address bits 15 - 0)
 *  bBaseAddressHighWordLowByte      = 0  - 7  (Base Address bits 23 - 16)
 *  bDescriptorFlags                 = 8  - 15 (Valid Flags =  Present Bit, DPL,System Bit, Type)
 *  bFlagsAndHighNibbleSegmentLength = 16 - 19 (High Segment Limit Bits 16-19)
 *                                   = 20 - 23 (Valid Flags = Available To OS bit, Enable Long Mode, DB, Granularity)
 *  bBaseAddressHighWordHighByte     = 24 - 31 (Base Address bits 24 - 31)
 **********************************************************/
#pragma pack(1)
typedef struct _GDT_SYSTEM_DESCRIPTOR_ENTRY
{
    unsigned short   wSegmentLength;
    unsigned short   wBaseAddressLowWord;
    unsigned char    bBaseAddressHighWordLowByte;
    unsigned char    bDescriptorFlags;
    unsigned char    bFlagsAndHighNibbleSegmentLength;
    unsigned char    bBaseAddressHighWordHighByte;

} GDT_SYSTEM_DESCRIPTOR_ENTRY, *PGDT_SYSTEM_DESCRIPTOR_ENTRY;

/**********************************************************
 * GDT Code Descriptor Entry (Legacy/Compatible Mode)
 *
 *    This is used for code. 
 *
 * 31              24  23  22  21   20  19    16  15 14 13  12  11 10  9   8   7         0
 * +-----------------+---+---+----+---+---------+---+-----+---+---+--+---+---+-----------+
 * | Base Address   | G | D  |NA | AVL| Limit  | P | DPL | 1 | 1 | C | R | A | Base Addr |
 * +------------------------------------------+----------------------------------------+
 * |        Base Address 15 - 0              |     Segment Limit 15 - 0                |
 * +----------------------------------------------------------------------------------+
 *
 *  wSegmentLength                   = 0  - 15 (Segment Limit bits 15 - 0)
 *  wBaseAddressLowWord              = 16 - 31 (Base Address bits 15 - 0)
 *  bBaseAddressHighWordLowByte      = 0  - 7  (Base Address bits 23 - 16)
 *  bDescriptorFlags                 = 8  - 15 (Valid Flags =  Present Bit, DPL,System Bit, Conforming bit, read/exe bit, accessed bit)
 *  bFlagsAndHighNibbleSegmentLength = 16 - 19 (High Segment Limit Bits 16-19)
 *                                   = 20 - 23 (Valid Flags = Available To OS bit, Enable Long Mode (avl only in long mode), Default Operand Size, Granularity)
 *  bBaseAddressHighWordHighByte     = 24 - 31 (Base Address bits 24 - 31)
 *
 *
 *
 * GDT Code Descriptor Entry (LONG MODE)
 *
 *    This is used for code descriptors in Longmode.  Majority of the descriptor table is not used.
 *    This is only in effect with the L bit is 1.  If the L bit is not 1 then the descriptor
 *    behaves in compatibility mode which was listed above. 
 *
 * 31                  23  22  21   20  19    16  15 14 13  12  11 10  9                 0
 * +---------------------+---+----+-------------+---+-----+-------+--+-------------------+
 * |                    | D  | L |             | P | DPL |       | C |                  |
 * +-----------------------------------------------------------------------------------+
 * |                                                                                   |
 * +----------------------------------------------------------------------------------+
 *
 *  wSegmentLength                   = 0 - 15  NOT USED
 *  wBaseAddressLowWord              = 16 - 31 NOT USED
 *  bBaseAddressHighWordLowByte      = 0  - 7  NOT USED
 *  bDescriptorFlags                 = 8  - 15 (Valid Flags =  Present Bit, DPL,Conforming bit)
 *  bFlagsAndHighNibbleSegmentLength = 16 - 19 NOT USED
 *                                   = 20 - 23 (Valid Flags = Enable Long Mode, Default Operand Size)
 *  bBaseAddressHighWordHighByte     = 24 - 31 NOT USED
 **********************************************************/
#pragma pack(1)
typedef struct _GDT_CODE_DESCRIPTOR_ENTRY
{
    unsigned short   wSegmentLength;
    unsigned short   wBaseAddressLowWord;
    unsigned char    bBaseAddressHighWordLowByte;
    unsigned char    bDescriptorFlags;
    unsigned char    bFlagsAndHighNibbleSegmentLength;
    unsigned char    bBaseAddressHighWordHighByte;

} GDT_CODE_DESCRIPTOR_ENTRY, *PGDT_CODE_DESCRIPTOR_ENTRY;


/**********************************************************
 * GDT Data Descriptor Entry (Legacy/Compatible Mode)
 *
 *    This is used for data. 
 *
 * 31              24  23  22  21   20  19    16  15 14 13  12  11 10  9   8   7         0
 * +-----------------+---+---+----+---+---------+---+-----+---+---+--+---+---+-----------+
 * | Base Address   | G | DB |NA | AVL| Limit  | P | DPL | 1 | 0 | E | W | A | Base Addr |
 * +------------------------------------------+----------------------------------------+
 * |        Base Address 15 - 0              |     Segment Limit 15 - 0                |
 * +----------------------------------------------------------------------------------+
 *
 *  wSegmentLength                   = 0  - 15 (Segment Limit bits 15 - 0)
 *  wBaseAddressLowWord              = 16 - 31 (Base Address bits 15 - 0)
 *  bBaseAddressHighWordLowByte      = 0  - 7  (Base Address bits 23 - 16)
 *  bDescriptorFlags                 = 8  - 15 (Valid Flags =  Present Bit, DPL,System Bit, Expand Down bit, writable bit, accessed bit)
 *  bFlagsAndHighNibbleSegmentLength = 16 - 19 (High Segment Limit Bits 16-19)
 *                                   = 20 - 23 (Valid Flags = Available To OS bit, Enable Long Mode (avl only in long mode), D/B (default operand size), Granularity)
 *  bBaseAddressHighWordHighByte     = 24 - 31 (Base Address bits 24 - 31)
 *
 *
 *
 * GDT data Descriptor Entry (LONG MODE)
 *
 *    This is used for data descriptors in Longmode.  Majority of the descriptor table is not used.
 *    
 *
 * 31                                         16  15 14                                 0
 * +--------------------------------------------+---+------------------------------------+
 * |                                           | P |                                    |
 * +-----------------------------------------------------------------------------------+
 * |                                                                                   |
 * +----------------------------------------------------------------------------------+
 *
 *  wSegmentLength                   = 0 - 15  NOT USED
 *  wBaseAddressLowWord              = 16 - 31 NOT USED
 *  bBaseAddressHighWordLowByte      = 0  - 7  NOT USED
 *  bDescriptorFlags                 = 8  - 15 (Valid Flags =  Present Bit)
 *  bFlagsAndHighNibbleSegmentLength = 16 - 23 NOT USED
 *  bBaseAddressHighWordHighByte     = 24 - 31 NOT USED
 **********************************************************/
#pragma pack(1)
typedef struct _GDT_DATA_DESCRIPTOR_ENTRY
{
    unsigned short   wSegmentLength;
    unsigned short   wBaseAddressLowWord;
    unsigned char    bBaseAddressHighWordLowByte;
    unsigned char    bDescriptorFlags;
    unsigned char    bFlagsAndHighNibbleSegmentLength;
    unsigned char    bBaseAddressHighWordHighByte;

} GDT_DATA_DESCRIPTOR_ENTRY, *PGDT_DATA_DESCRIPTOR_ENTRY;



NTSTATUS GDT_ReadSelectorInformationFromGDT(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);
NTSTATUS GDT_Free64BitGlobalDescriptorTable(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);
NTSTATUS GDT_Setup64BitGlobalDescriptorTable(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);

unsigned short GDT_CreateLongModeDescriptor(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);
unsigned short GDT_CreateCodeIdentityDescriptor(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress);
NTSTATUS GDT_CreateCopyOfDescriptorTable(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);
unsigned short GDT_CreateDataIdentityDescriptor(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress);

#endif






