/**********************************************************************
 * 
 *  Toby Opferman
 *
 *  Mutiny Driver!
 *
 *  This driver implements Operating System Mutiny!
 *  
 *  This example is for educational purposes only.  I license this source
 *  out for use in learning how to write a device driver.
 *
 *     Driver Entry Point  Copyright (c) 2005, All Rights Reserved
 **********************************************************************/


#define _X86_

#include <ddk/ntddk.h>
#include "privatedata.h"

#define LINEAR_ADDRESS_TO_PDP_INDEX_PAE(x) ((x>>30) & 0x3)
#define LINEAR_ADDRESS_TO_PDE_INDEX_PAE(x) ((x>>21) & 0x1FF)
#define LINEAR_ADDRESS_TO_PTE_INDEX_PAE(x) ((x>>12) & 0x1FF)


#define PHYSICAL_ADDRESS_TO_4K_BOUNDARY(h, l, a) \
               l = ((a>>8) & 0xFFF0);  \
               h = ((a>>24) & 0xFF);
               
#define PHYSICAL_ADDRESS_TO_4K_BOUNDARY_PAE_STYLE(h, l, a) \
               l = ((a>>8) & 0xFFFFFFF0);  \
               h = (0);   /* Not Used, only 36 bits */
               



#define PAE_PAGE_POOL  40
               
                       
/*
 * Internal Functions
 */
PPAGETABLE_PTE_ENTRY_PAE32 PhysicalAddressExtensions32_AllocatePageDirectoryEntry(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiPageDirectoryIndex, PPAGETABLE_PDE_ENTRY_PAE32 pPageDirectoryEntry);
PPAGETABLE_PDE_ENTRY_PAE32 PhysicalAddressExtensions32_AllocatePageDirectoryPointer(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiPageDirectoryPointerIndex);

/**********************************************************************
 * 
 *  PhysicalAddressExtensions32_CreatePageTables
 *
 *    This API will Create a page table 
 *    in 32 bit PAE
 *
 **********************************************************************/
NTSTATUS PhysicalAddressExtensions32_CreatePageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{   
    NTSTATUS NtStatus = STATUS_SUCCESS;
     
    /*
     * We are going to pray that the Operating System will allocate our pages on 4k boundaries
     * so we don't have wasted bytes.  We need 1 Page for PDE, then 1 Page PER PTE unless we
     * do 4 Megabyte boundaries (which isn't a bad idea).
     *                                                     
     *
     */

    NtStatus = Memory_AllocateNonPaged4kPhysicallyAligned(PAE_PAGE_POOL, &pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock);
                                       
    if(NT_SUCCESS(NtStatus))
    {
        PHYSICAL_ADDRESS_TO_4K_BOUNDARY(pOperatingSystemThunkState->PAEPageTables32.CR3_32bit.bTopPhysicalAddress, pOperatingSystemThunkState->PAEPageTables32.CR3_32bit.wPhysicalAddressLowNibbleReserved, pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress);
            
        memset(pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.pAllocatedVirtualMemory, 0, pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.uiVirtualBlockSize);
        
        DbgPrint("32 bit Physical Address Extensions Page Tables Created\n");
        DbgPrint("   Allocated Virtual Address %0x\n", pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.pAllocatedVirtualMemory);
        DbgPrint("   Allocated Physical Address %0x\n", pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.PhysicalAddress);
        DbgPrint("     Block Size %i\n", pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.uiVirtualBlockSize);
        DbgPrint("   Adjusted Virtual Address %0x\n", pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress);
        DbgPrint("   Adjusted Physical Address %0x\n", pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress);
        DbgPrint("     Block Size %i\n", pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.uiPageAlignedBlockSize);
        DbgPrint("  CR3 = %0x%0x00\n\n", pOperatingSystemThunkState->PAEPageTables32.CR3_32bit.bTopPhysicalAddress, pOperatingSystemThunkState->PAEPageTables32.CR3_32bit.wPhysicalAddressLowNibbleReserved);
    }

    return NtStatus;
}


/**********************************************************************
 * 
 *  PhysicalAddressExtensions32_DestroyPageTables
 *
 *    This API will Destroy a page table 
 *    in 32 bit PAE
 *
 **********************************************************************/
NTSTATUS PhysicalAddressExtensions32_DestroyPageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{
    NTSTATUS NtStatus = STATUS_SUCCESS;

    Memory_FreeNonPaged4kAligned(&pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock);

    return NtStatus;
}




/**********************************************************************
 * 
 *  PhysicalAddressExtensions32_MapVirtualToPhysical
 *
 *    This API will map a virtual address to a physical address
 *    in 32 bit in PAE
 *
 **********************************************************************/
NTSTATUS PhysicalAddressExtensions32_MapVirtualToPhysical(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress, unsigned int  uiVirtualAddress, unsigned int  uiPhysicalAddress)
{   
    NTSTATUS NtStatus = STATUS_SUCCESS;
    unsigned int uiLinearAddress = uiBaseAddress + uiVirtualAddress;
    unsigned int uiPageDirectoryPointerIndex = 0, uiPageDirectoryIndex = 0, uiPageTableIndex = 0;
    unsigned int uiAvailablePhysicalAddress = 0;
    BOOLEAN bFoundAvailablePhysicalPage = FALSE;
    PPAGETABLE_PTE_ENTRY_PAE32 pPageTableEntry;
    PPAGETABLE_PDE_ENTRY_PAE32 pPageDirectoryEntry;
    PHYSICAL_ADDRESS PhysicalAddress;
    unsigned short usHighAddress;
    unsigned int uiLowAddress;

    uiPageDirectoryPointerIndex  = LINEAR_ADDRESS_TO_PDP_INDEX_PAE(uiLinearAddress);
    uiPageDirectoryIndex         = LINEAR_ADDRESS_TO_PDE_INDEX_PAE(uiLinearAddress);
    uiPageTableIndex             = LINEAR_ADDRESS_TO_PTE_INDEX_PAE(uiLinearAddress);

    DbgPrint("MapVirtualToPhysical in PAE\n");
    DbgPrint("   Base Descriptor Address = %0x\n", uiBaseAddress);
    DbgPrint("   Base Virtual Address = %0x\n", uiVirtualAddress);
    DbgPrint("   Adjusted Linear Address = %0x\n",uiLinearAddress);
    DbgPrint("   Desired Physical Address = %0x\n", uiPhysicalAddress);
    DbgPrint("   Page Directory Pointer Index = %0x, Page Directory Index = %0x, Page Table Index = %0x\n", uiPageDirectoryPointerIndex, uiPageDirectoryIndex, uiPageTableIndex);
    
    pPageDirectoryEntry = PhysicalAddressExtensions32_AllocatePageDirectoryPointer(pOperatingSystemThunkState, uiPageDirectoryPointerIndex);

    if(pPageDirectoryEntry)
    {

        pPageTableEntry = PhysicalAddressExtensions32_AllocatePageDirectoryEntry(pOperatingSystemThunkState, uiPageDirectoryIndex, pPageDirectoryEntry);

        if(pPageTableEntry)
        {

            if(pPageTableEntry[uiPageTableIndex].dwPhysicalAddressAndAvailableBits == 0 && pPageTableEntry[uiPageTableIndex].wPhysicalAddress == 0)
            {
               pPageTableEntry[uiPageTableIndex].bPagingAttributes = PAGING_PAE_FLAG_IS_PRESENT | PAGING_PAE_FLAG_WRITABLE;
               PHYSICAL_ADDRESS_TO_4K_BOUNDARY_PAE_STYLE(pPageTableEntry[uiPageTableIndex].wPhysicalAddress, pPageTableEntry[uiPageTableIndex].dwPhysicalAddressAndAvailableBits, uiPhysicalAddress);
            }
            else
            {
                DbgPrint("  Pages Already Mapped \n");
                PHYSICAL_ADDRESS_TO_4K_BOUNDARY_PAE_STYLE(usHighAddress, uiLowAddress, uiPhysicalAddress);
                
                if(pPageTableEntry[uiPageTableIndex].wPhysicalAddress != usHighAddress || pPageTableEntry[uiPageTableIndex].dwPhysicalAddressAndAvailableBits != uiLowAddress)
                {
                    DbgPrint("  ********** Mapping Conflict!!! %0x%0x00 != %0x%0x00\n", pPageTableEntry[uiPageTableIndex].wPhysicalAddress, pPageTableEntry[uiPageTableIndex].dwPhysicalAddressAndAvailableBits, usHighAddress, uiLowAddress);
                    _asm int 3
                }
    
                NtStatus = STATUS_UNSUCCESSFUL;
            }

            DbgPrint("   Index Page Table Virtual Address -> %0x\n", &pPageTableEntry[uiPageTableIndex]);
            PhysicalAddress = MmGetPhysicalAddress(&pPageTableEntry[uiPageTableIndex]);
            DbgPrint("   Index Page Table Physical Address -> %0x\n", (unsigned int)PhysicalAddress.QuadPart);
            DbgPrint("   Mapped To Physical Address %0x%0x00\n", pPageTableEntry[uiPageTableIndex].wPhysicalAddress, pPageTableEntry[uiPageTableIndex].dwPhysicalAddressAndAvailableBits);
        }
        else
        {

            DbgPrint("Cannot find Page Table Entry\n");
            NtStatus = STATUS_UNSUCCESSFUL;
        }
    }
    else
    {
        DbgPrint("Cannot find Page Directory Entry\n");
        NtStatus = STATUS_UNSUCCESSFUL;
    }

    return NtStatus;
}


/**********************************************************************
 * 
 *  PhysicalAddressExtensions32_AllocatePageDirectoryPointer
 *
 *    This API will Allocate a Page Directory Pointer
 *
 **********************************************************************/
PPAGETABLE_PDE_ENTRY_PAE32 PhysicalAddressExtensions32_AllocatePageDirectoryPointer(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiPageDirectoryPointerIndex)
{
    NTSTATUS NtStatus = STATUS_SUCCESS;
    PPAGETABLE_PDE_ENTRY_PAE32 pPageDirectoryEntry = (PPAGETABLE_PDE_ENTRY_PAE32)pOperatingSystemThunkState->PAEPageTables32.pdePageDirectoryPointerIndex[uiPageDirectoryPointerIndex];
    PPAGETABLE_PDP_ENTRY_PAE32 pPageDirectoryPointers = (PPAGETABLE_PDP_ENTRY_PAE32)pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress;
    PHYSICAL_ADDRESS PhysicalAddress = {0};
    PVOID pPhysicalAdddressOfNewPageDirectoryTable = 0;
    PVOID pVirtualAdddressOfNewPageDirectoryTable = 0;
    
    if(pPageDirectoryEntry)
    {
       DbgPrint("  Existing Page Directory Table Mapping (PDP Index %0x) -> Page Directory %0x\n", uiPageDirectoryPointerIndex, pPageDirectoryEntry);
    }
    else
    {
       pOperatingSystemThunkState->PAEPageTables32.uiLastPageIndexAllocated++;

       pPhysicalAdddressOfNewPageDirectoryTable = (PVOID)((unsigned char *)pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress + (pOperatingSystemThunkState->PAEPageTables32.uiLastPageIndexAllocated<<12));
       pVirtualAdddressOfNewPageDirectoryTable  = (PVOID)((unsigned char *)pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress + (pOperatingSystemThunkState->PAEPageTables32.uiLastPageIndexAllocated<<12));

       PHYSICAL_ADDRESS_TO_4K_BOUNDARY_PAE_STYLE(pPageDirectoryPointers[uiPageDirectoryPointerIndex].wPhysicalAddress, pPageDirectoryPointers[uiPageDirectoryPointerIndex].dwPhysicalAddressAndAvailableBits, (unsigned int)pPhysicalAdddressOfNewPageDirectoryTable);
       pPageDirectoryPointers[uiPageDirectoryPointerIndex].bPagingAttributes = 1;

       pPageDirectoryEntry = pOperatingSystemThunkState->PAEPageTables32.pdePageDirectoryPointerIndex[uiPageDirectoryPointerIndex] = (PPAGETABLE_PDE_ENTRY_PAE32)pVirtualAdddressOfNewPageDirectoryTable;

       /*
        * For Debugging Purposes
        */
       
       DbgPrint("  Creating new PDP Association PDE\n");
       DbgPrint("     PDP(Index %0x)\n", uiPageDirectoryPointerIndex);
       DbgPrint("     Virtual Address Of PDP Entry(%0x)\n", &pPageDirectoryPointers[uiPageDirectoryPointerIndex]);
       PhysicalAddress = MmGetPhysicalAddress(&pPageDirectoryPointers[uiPageDirectoryPointerIndex]);
       DbgPrint("     Physical Address Of PDP Entry(%0x)\n", (unsigned int)PhysicalAddress.QuadPart);
       DbgPrint("     Virtual Address Of New Page Directory Entry(%0x)\n", pVirtualAdddressOfNewPageDirectoryTable);
       PhysicalAddress = MmGetPhysicalAddress(pVirtualAdddressOfNewPageDirectoryTable);
       DbgPrint("     Physical Address Of New Page Directory Entry(%0x), What we said it was (%0x)\n", (unsigned int)PhysicalAddress.QuadPart, pPhysicalAdddressOfNewPageDirectoryTable);
       DbgPrint("     PDP Entry = (%0x%0x00)\n", pPageDirectoryPointers[uiPageDirectoryPointerIndex].wPhysicalAddress, pPageDirectoryPointers[uiPageDirectoryPointerIndex].dwPhysicalAddressAndAvailableBits);
    }      
    
    return pPageDirectoryEntry;
}

/**********************************************************************
 * 
 *  PhysicalAddressExtensions32_AllocatePageDirectoryEntry
 *
 *    This API will Allocate a Page Directory Entry
 *
 **********************************************************************/
PPAGETABLE_PTE_ENTRY_PAE32 PhysicalAddressExtensions32_AllocatePageDirectoryEntry(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiPageDirectoryIndex, PPAGETABLE_PDE_ENTRY_PAE32 pPageDirectoryEntry)
{
    NTSTATUS NtStatus = STATUS_SUCCESS;
    PPAGETABLE_PTE_ENTRY_PAE32 pPageTableEntry = NULL;
    PHYSICAL_ADDRESS PhysicalAddress = {0};
    PVOID pPhysicalAdddressOfNewPageTable = 0;
    PVOID pVirtualAdddressOfNewPageTable = 0;
    unsigned int uiPhysicalToVirtualOffset = 0;
    
    if(pPageDirectoryEntry[uiPageDirectoryIndex].wPhysicalAddress != 0 || pPageDirectoryEntry[uiPageDirectoryIndex].dwPhysicalAddressAndAvailableBits != 0)
    {
        uiPhysicalToVirtualOffset = (pPageDirectoryEntry[uiPageDirectoryIndex].dwPhysicalAddressAndAvailableBits & 0xFFFFFFF0)<<8;
        uiPhysicalToVirtualOffset = uiPhysicalToVirtualOffset -   pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress;
        pPageTableEntry = (PPAGETABLE_PTE_ENTRY_PAE32)((unsigned char *)pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress + uiPhysicalToVirtualOffset);
       
       DbgPrint("  Existing PDE Mapping (Index %0x) -> Page Table %0x\n", uiPageDirectoryIndex, pPageTableEntry);
    }
    else
    {
       pOperatingSystemThunkState->PAEPageTables32.uiLastPageIndexAllocated++;

       pPhysicalAdddressOfNewPageTable = (PVOID)((unsigned char *)pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress + (pOperatingSystemThunkState->PAEPageTables32.uiLastPageIndexAllocated<<12));
       pVirtualAdddressOfNewPageTable  = (PVOID)((unsigned char *)pOperatingSystemThunkState->PAEPageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress + (pOperatingSystemThunkState->PAEPageTables32.uiLastPageIndexAllocated<<12));

       PHYSICAL_ADDRESS_TO_4K_BOUNDARY_PAE_STYLE(pPageDirectoryEntry[uiPageDirectoryIndex].wPhysicalAddress, pPageDirectoryEntry[uiPageDirectoryIndex].dwPhysicalAddressAndAvailableBits, (unsigned int)pPhysicalAdddressOfNewPageTable);
       pPageDirectoryEntry[uiPageDirectoryIndex].bPagingAttributes = PAGING_FLAG_IS_PRESENT32 | PAGING_FLAG_WRITABLE32;

       pPageTableEntry  = (PPAGETABLE_PTE_ENTRY_PAE32)pVirtualAdddressOfNewPageTable;

       /*
        * For Debugging Purposes
        */
       
       DbgPrint("  Creating new PDE Association PTE\n");
       DbgPrint("     PDE(Index %0x)\n", uiPageDirectoryIndex);
       DbgPrint("     Virtual Address Of PDE Entry(%0x)\n", &pPageDirectoryEntry[uiPageDirectoryIndex]);
       PhysicalAddress = MmGetPhysicalAddress(&pPageDirectoryEntry[uiPageDirectoryIndex]);
       DbgPrint("     Physical Address Of PDE Entry(%0x)\n", (unsigned int)PhysicalAddress.QuadPart);
       DbgPrint("     Virtual Address Of New Page Table(%0x)\n", pVirtualAdddressOfNewPageTable);
       PhysicalAddress = MmGetPhysicalAddress(pVirtualAdddressOfNewPageTable);
       DbgPrint("     Physical Address Of New Page Table(%0x), What we said it was (%0x)\n", (unsigned int)PhysicalAddress.QuadPart, pPhysicalAdddressOfNewPageTable);
       DbgPrint("     PDE Entry = (%0x%0x00)\n", pPageDirectoryEntry[uiPageDirectoryIndex].wPhysicalAddress, pPageDirectoryEntry[uiPageDirectoryIndex].dwPhysicalAddressAndAvailableBits);
    }      
    
    return pPageTableEntry;
}


