/*******************************************************
 *
 *  Tile.c
 *
 *     Tile Engine
 *
 *  Copyright (c) 2003, All Rights Reserved
 *
 *******************************************************/
 
 
 
#include <windows.h>
#include <tile.h>
#include <stdarg.h>
#include <stdio.h>

 
/*********************************************************
 * Internal Functions
 *********************************************************/
 typedef struct _TILE_SPRITE_ENG;

 typedef struct {

    PVOID pTileContext;
    PFNDRAWTILE pfnDrawTile;
	PFNSPRITECALLBACK pfnDrawSprite;

	DWORD dwResolutionX;
	DWORD dwResolutionY;

	DWORD dwTileSizeX;
	DWORD dwTileSizeY;

	DWORD dwCurrentTileX;
	DWORD dwCurrentTileY;

	DWORD dwCurrentTileIndexX;
	DWORD dwCurrentTileIndexY;

    DWORD dwCurrentViewWidth;
	DWORD dwCurrentViewHeight;

	TILE_SCREEN_POSITION TileScreenPosition;

    PTILEINFO pTileMap;

	struct _TILE_SPRITE_ENG *pSprites;

 } TILEENG, *PTILEENG;


 typedef struct _TILE_SPRITE_ENG
{
	DWORD dwSpriteId;
    PVOID             pTileContext;
    PFNSPRITECALLBACK pfnDrawSprite;
	PFNSPRITECOLLISION pfnSpriteCollision;

	DWORD dwCurrentTileX;
	DWORD dwCurrentTileY;

	DWORD dwViewWidth;
	DWORD dwViewHeight;

	DWORD dwFlags;

	PTILEENG pTileEng;

	struct _TILE_SPRITE_ENG *pNextSprite;
	struct _TILE_SPRITE_ENG *pPrevSprite;

} TILE_SPRITE_ENG, *PTILE_SPRITE_ENG;



void Tile_Debug(char *pszFormatString, ...);
DWORD Tile_FindTileStartPosition(DWORD dwTileIndex, DWORD dwCurrTile, DWORD dwTileResolution, DWORD dwTileSize, DWORD dwScreenResolution, DWORD *pdwStart, DWORD *pdwCurrentTile); 

 /********************************************************
  *  Tile_Init
  *
  *     Initialize The Tile Engine
  *   
  *
  *
  ********************************************************/
HTILE WINAPI Tile_Init(PTILE_MAP pTileMap)
{
	PTILEENG pTileEng = NULL;

	if(pTileEng = (PTILEENG)LocalAlloc(LMEM_ZEROINIT, sizeof(TILEENG)))
	{
		pTileEng->dwResolutionX = pTileMap->dwResolutionX;
		pTileEng->dwResolutionY = pTileMap->dwResolutionY;
		pTileEng->dwTileSizeX   = pTileMap->dwTileSizeX;
		pTileEng->dwTileSizeY   = pTileMap->dwTileSizeY;
		pTileEng->pfnDrawTile   = pTileMap->pfnDrawTile;
        pTileEng->pTileContext  = pTileMap->pTileContext;
		pTileEng->pfnDrawSprite = pTileMap->pfnDrawSprite;

		pTileEng->dwCurrentTileX  = pTileMap->dwCurrentTileX;
		pTileEng->dwCurrentTileY  = pTileMap->dwCurrentTileY;

		pTileEng->dwCurrentViewWidth  = pTileMap->dwViewWidth;
		pTileEng->dwCurrentViewHeight = pTileMap->dwViewHeight;

		if(pTileEng->pTileMap = (PTILEINFO)LocalAlloc(LMEM_ZEROINIT, sizeof(TILEINFO)*pTileEng->dwResolutionY*pTileEng->dwResolutionX))
		{
			memcpy(pTileEng->pTileMap, pTileMap->pTileInfo, sizeof(TILEINFO)*pTileEng->dwResolutionY*pTileEng->dwResolutionX);
		}
		else
		{
			Tile_UnInit((HTILE)pTileEng);
			pTileEng = NULL;
		}
	}

    return (HTILE)pTileEng;
}


 /********************************************************
  *  Tile_UnInit
  *
  *     Free the tile context
  *   
  *
  *
  ********************************************************/
void WINAPI Tile_UnInit(HTILE hTile)
{
	PTILEENG pTileEng = (PTILEENG)hTile;

	if(pTileEng)
	{
		if(pTileEng->pTileMap)
		{
			LocalFree(pTileEng->pTileMap);
		}

		LocalFree(pTileEng);
	}

}

 /********************************************************
  *  Tile_MoveUp
  *
  *     Move the current position up.
  *   
  *
  *
  ********************************************************/
BOOL WINAPI Tile_MoveUp(HTILE hTile, DWORD dwPosition)
{
	PTILEENG pTileEng = (PTILEENG)hTile;
    BOOL bDoneMoving  = FALSE;
	BOOL bNextTileIsBlocked = TRUE;
    
    
	while(!bDoneMoving)
	{

		bNextTileIsBlocked = TRUE;

		if(pTileEng->dwCurrentTileY != 0)
		{
			if((pTileEng->pTileMap[pTileEng->dwCurrentTileX + (pTileEng->dwCurrentTileY -1)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
			{
				if(pTileEng->dwTileSizeX + (pTileEng->dwTileSizeX/10) > pTileEng->dwCurrentTileIndexX + pTileEng->dwCurrentViewWidth)
				{
					bNextTileIsBlocked = FALSE;
				}
				else
				{
					if(pTileEng->dwCurrentTileX != (pTileEng->dwResolutionX - 1))
					{
						if((pTileEng->pTileMap[pTileEng->dwCurrentTileX + 1 + (pTileEng->dwCurrentTileY - 1)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
						{
							bNextTileIsBlocked = FALSE;
						}
					}
				}
			}
		}
		

		if(dwPosition > pTileEng->dwCurrentTileIndexY)
		{
			if(pTileEng->dwCurrentTileY == 0 || bNextTileIsBlocked)
			{
				pTileEng->dwCurrentTileIndexY = 0;
				bDoneMoving = TRUE;
			}
			else
			{
				pTileEng->dwCurrentTileY -= 1;
				dwPosition -= pTileEng->dwCurrentTileIndexY;
				pTileEng->dwCurrentTileIndexY = pTileEng->dwTileSizeY - 1;
			}
		}
		else
		{
			if(dwPosition == pTileEng->dwCurrentTileIndexY)
			{
				if(pTileEng->dwCurrentTileY == 0 || bNextTileIsBlocked)
				{
					pTileEng->dwCurrentTileIndexY = 0;
				}
				else
				{
					pTileEng->dwCurrentTileY         -= 1;
					pTileEng->dwCurrentTileIndexY = pTileEng->dwTileSizeY - 1;
				}
	
			}
			else
			{
				pTileEng->dwCurrentTileIndexY -= dwPosition; 
			}

			bDoneMoving = TRUE;
		}
	}

    
	return TRUE;
}

 /********************************************************
  *  Tile_MoveDown
  *
  *     Move the current position down.
  *   
  *
  *
  ********************************************************/
BOOL WINAPI Tile_MoveDown(HTILE hTile, DWORD dwPosition)
{
	PTILEENG pTileEng = (PTILEENG)hTile;
	DWORD dwTileLength;
    BOOL bDoneMoving  = FALSE;
	BOOL bNextTileIsBlocked;
	
	while(!bDoneMoving)
	{
		dwTileLength = (pTileEng->dwTileSizeY - pTileEng->dwCurrentTileIndexY);
		bNextTileIsBlocked = TRUE;

		if(pTileEng->dwCurrentTileY != (pTileEng->dwResolutionY - 1))
		{
			if((pTileEng->pTileMap[pTileEng->dwCurrentTileX + (pTileEng->dwCurrentTileY + 1)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
			{
				if(pTileEng->dwTileSizeX + (pTileEng->dwTileSizeX/10) > pTileEng->dwCurrentTileIndexX + pTileEng->dwCurrentViewWidth)
				{
					bNextTileIsBlocked = FALSE;
				}
				else
				{
					if(pTileEng->dwCurrentTileX != (pTileEng->dwResolutionX - 1))
					{
						if((pTileEng->pTileMap[pTileEng->dwCurrentTileX + 1 + (pTileEng->dwCurrentTileY + 1)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
						{
							bNextTileIsBlocked = FALSE;
						}
					}
				}
			}
		}
		

		if(dwPosition > dwTileLength)
		{
			if(pTileEng->dwCurrentTileY == (pTileEng->dwResolutionY - 1) || bNextTileIsBlocked)
			{
				pTileEng->dwCurrentTileIndexY = pTileEng->dwTileSizeY - pTileEng->dwCurrentViewHeight;
				bDoneMoving = TRUE;
			}
			else
			{
				pTileEng->dwCurrentTileY += 1;
				dwPosition -= dwTileLength;
				pTileEng->dwCurrentTileIndexY = 0;
			}
		}
		else
		{
			if(dwPosition == dwTileLength)
			{
				if(pTileEng->dwCurrentTileY == (pTileEng->dwResolutionY - 1) || bNextTileIsBlocked)
				{
					pTileEng->dwCurrentTileIndexY = pTileEng->dwTileSizeY - pTileEng->dwCurrentViewHeight;
				}
				else
				{
					pTileEng->dwCurrentTileY += 1;
					pTileEng->dwCurrentTileIndexY = 0;
				}
	
			}
			else
			{
				if(pTileEng->dwCurrentTileIndexY + dwPosition + pTileEng->dwCurrentViewHeight >= dwTileLength && 
					((pTileEng->dwCurrentTileY >= (pTileEng->dwResolutionY - 1)) || bNextTileIsBlocked))
				{
					pTileEng->dwCurrentTileIndexY = pTileEng->dwTileSizeY - pTileEng->dwCurrentViewHeight;

				}
				else
				{
					pTileEng->dwCurrentTileIndexY += dwPosition; 
				}
			}

			bDoneMoving = TRUE;
		}
	}
	
	return TRUE;
}

 /********************************************************
  *  Tile_MoveLeft
  *
  *     Move the current position left.
  *   
  *
  *
  ********************************************************/
BOOL WINAPI Tile_MoveLeft(HTILE hTile, DWORD dwPosition)
{
	PTILEENG pTileEng = (PTILEENG)hTile;
    BOOL bDoneMoving  = FALSE;
	BOOL bNextTileIsBlocked;

    while(!bDoneMoving)
	{

		bNextTileIsBlocked = TRUE;

		if(pTileEng->dwCurrentTileX != 0)
		{
			if((pTileEng->pTileMap[pTileEng->dwCurrentTileX - 1 + (pTileEng->dwCurrentTileY)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
			{
				if(pTileEng->dwTileSizeY + (pTileEng->dwTileSizeY/10) > pTileEng->dwCurrentTileIndexY + pTileEng->dwCurrentViewHeight)
				{
					bNextTileIsBlocked = FALSE;
				}
				else
				{
					if(pTileEng->dwCurrentTileY != (pTileEng->dwResolutionY - 1))
					{
						if((pTileEng->pTileMap[pTileEng->dwCurrentTileX - 1 + (pTileEng->dwCurrentTileY + 1)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
						{
							bNextTileIsBlocked = FALSE;
						}
					}
				}
			}
		}
		

		if(dwPosition > pTileEng->dwCurrentTileIndexX)
		{
			if(pTileEng->dwCurrentTileX == 0 || bNextTileIsBlocked)
			{
				pTileEng->dwCurrentTileIndexX = 0;
				bDoneMoving = TRUE;
			}
			else
			{
				pTileEng->dwCurrentTileX -= 1;
				dwPosition -= pTileEng->dwCurrentTileIndexX;
				pTileEng->dwCurrentTileIndexX = pTileEng->dwTileSizeX - 1;
			}
		}
		else
		{
			if(dwPosition == pTileEng->dwCurrentTileIndexX)
			{
				if(pTileEng->dwCurrentTileX == 0 || bNextTileIsBlocked)
				{
					pTileEng->dwCurrentTileIndexX = 0;
				}
				else
				{
					pTileEng->dwCurrentTileX         -= 1;
					pTileEng->dwCurrentTileIndexX = pTileEng->dwTileSizeX - 1;
				}
	
			}
			else
			{
				pTileEng->dwCurrentTileIndexX -= dwPosition; 
			}

			bDoneMoving = TRUE;
		}
	}
	
	return TRUE;

}

 /********************************************************
  *  Tile_MoveRight
  *
  *     Move the current position right.
  *   
  *
  *
  ********************************************************/
BOOL WINAPI Tile_MoveRight(HTILE hTile, DWORD dwPosition)
{
	PTILEENG pTileEng = (PTILEENG)hTile;
	DWORD dwTileLength;
    BOOL bDoneMoving  = FALSE;
	BOOL bNextTileIsBlocked = TRUE;
	
	while(!bDoneMoving)
	{
		bNextTileIsBlocked = TRUE;

		if(pTileEng->dwCurrentTileX != (pTileEng->dwResolutionX - 1))
		{
			if((pTileEng->pTileMap[pTileEng->dwCurrentTileX + 1 + (pTileEng->dwCurrentTileY)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
			{
				if(pTileEng->dwTileSizeY + (pTileEng->dwTileSizeY/10) > pTileEng->dwCurrentTileIndexY + pTileEng->dwCurrentViewHeight)
				{
					bNextTileIsBlocked = FALSE;
				}
				else
				{
					if(pTileEng->dwCurrentTileY != (pTileEng->dwResolutionY - 1))
					{
						if((pTileEng->pTileMap[pTileEng->dwCurrentTileX + 1 + (pTileEng->dwCurrentTileY + 1)*pTileEng->dwResolutionX].dwTileFlags & TF_BLOCKED) == 0)
						{
							bNextTileIsBlocked = FALSE;
						}
					}
				}
			}
		}


		dwTileLength = (pTileEng->dwTileSizeX - pTileEng->dwCurrentTileIndexX);

		if(dwPosition > dwTileLength)
		{
			if(pTileEng->dwCurrentTileX == (pTileEng->dwResolutionX - 1) || bNextTileIsBlocked)
			{
				pTileEng->dwCurrentTileIndexX = pTileEng->dwTileSizeX - pTileEng->dwCurrentViewWidth;;
				bDoneMoving = TRUE;
			}
			else
			{
				pTileEng->dwCurrentTileX += 1;
				dwPosition -= dwTileLength;
				pTileEng->dwCurrentTileIndexX = 0;
			}
		}
		else
		{
			if(dwPosition == dwTileLength)
			{
				if(pTileEng->dwCurrentTileX == (pTileEng->dwResolutionX - 1) || bNextTileIsBlocked)
				{
					pTileEng->dwCurrentTileIndexX = pTileEng->dwTileSizeX - pTileEng->dwCurrentViewWidth;;
				}
				else
				{
					pTileEng->dwCurrentTileX += 1;
					pTileEng->dwCurrentTileIndexX = 0;
				}
	
			}
			else
			{
				if(pTileEng->dwCurrentTileIndexX + dwPosition + pTileEng->dwCurrentViewWidth >= dwTileLength && 
					((pTileEng->dwCurrentTileX >= (pTileEng->dwResolutionX - 1)) || bNextTileIsBlocked))
				{
					pTileEng->dwCurrentTileIndexX = pTileEng->dwTileSizeX - pTileEng->dwCurrentViewWidth;
				}
				else
				{
					pTileEng->dwCurrentTileIndexX += dwPosition; 
				}
			}

			bDoneMoving = TRUE;
		}
	}
	
	return TRUE;

}

 /********************************************************
  *  Tile_Draw
  *
  *     Draw the tiles onto the screen
  *   
  *
  *
  ********************************************************/
void WINAPI Tile_Draw(HTILE hTile, UINT dwMaxX, UINT dwMaxY)
{
	PTILEENG pTileEng = (PTILEENG)hTile;
	DWORD dwTilesX, dwTilesY, dwStartX, dwStartY;
	DWORD dwIndexX, dwIndexY, dwScreenX, dwScreenY;
	DWORD dwCurrentTileX, dwCurrentTileY;
    PTILEINFO pTileMap;

	dwTilesX = Tile_FindTileStartPosition(pTileEng->dwCurrentTileIndexX, pTileEng->dwCurrentTileX, pTileEng->dwResolutionX, pTileEng->dwTileSizeX, dwMaxX, &dwStartX, &dwCurrentTileX); 
	dwTilesY = Tile_FindTileStartPosition(pTileEng->dwCurrentTileIndexY, pTileEng->dwCurrentTileY, pTileEng->dwResolutionY, pTileEng->dwTileSizeY, dwMaxY, &dwStartY, &dwCurrentTileY); 

    for(dwIndexY = 0; dwIndexY < dwTilesY; dwIndexY++)
	{
		for(dwIndexX = 0; dwIndexX < dwTilesX; dwIndexX++)
		{
			pTileMap = &pTileEng->pTileMap[dwCurrentTileX + dwIndexX + (dwCurrentTileY + dwIndexY)*pTileEng->dwResolutionX];

            dwScreenX = dwIndexX ? ((pTileEng->dwTileSizeX - dwStartX) + (dwIndexX - 1)*pTileEng->dwTileSizeX) : 0;
		    dwScreenY = dwIndexY ? ((pTileEng->dwTileSizeY - dwStartY) + (dwIndexY - 1)*pTileEng->dwTileSizeY) : 0;

			if(pTileEng->dwCurrentTileY == (dwCurrentTileY + dwIndexY) && pTileEng->dwCurrentTileX == (dwCurrentTileX + dwIndexX))
			{
				pTileEng->TileScreenPosition.ScreenX = dwScreenX + pTileEng->dwCurrentTileIndexX;
				pTileEng->TileScreenPosition.ScreenY = dwScreenY + pTileEng->dwCurrentTileIndexY;
			}
            
            pTileEng->pfnDrawTile(pTileEng->pTileContext, pTileMap->dwTileId, dwScreenX, dwScreenY, (dwIndexX ? 0 : dwStartX), (dwIndexY ? 0 : dwStartY));
		}
	}

	pTileEng->pfnDrawSprite(pTileEng->pTileContext, 0, pTileEng->TileScreenPosition.ScreenX, pTileEng->TileScreenPosition.ScreenY);
}


/***********************************************************************
 * Tile_Modify
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
void WINAPI Tile_Modify(HTILE hTile, UINT TileX, UINT TileY, PTILEINFO pNewTileInfo)
{
	PTILEENG pTileEng = (PTILEENG)hTile;

	pTileEng->pTileMap[TileX + pTileEng->dwResolutionX*TileY] = *pNewTileInfo;
}



/***********************************************************************
 * Tile_GetCurrentTileLocation
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
void WINAPI Tile_GetCurrentTileLocation(HTILE hTile, PTILE_SCREEN_POSITION pTileScreenPosition)
{
	PTILEENG pTileEng = (PTILEENG)hTile;

	*pTileScreenPosition = pTileEng->TileScreenPosition;
}


/***********************************************************************
 * Tile_FindTileStartPosition
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
DWORD Tile_FindTileStartPosition(DWORD dwTileIndex, DWORD dwCurrTile, DWORD dwTileResolution, DWORD dwTileSize, DWORD dwScreenResolution, DWORD *pdwStart, DWORD *pdwCurrentTile)
{
	DWORD dwTilesToDraw = 0;
    DWORD dwHalfResolution;
	DWORD dwTempScreenRes;
	int iCurrentTile = dwCurrTile;

	dwHalfResolution = dwScreenResolution/2;
    dwHalfResolution -= dwTileIndex;
    
	*pdwStart = 0;

	do {
        
		if(dwHalfResolution >= dwTileSize)
		{
           dwHalfResolution -= dwTileSize;
		   iCurrentTile--;
		}
		else
		{
		   *pdwStart = dwTileSize - dwHalfResolution;
           iCurrentTile--;
		   dwHalfResolution = 0;
		}
		

	} while(dwHalfResolution);
    
	if(iCurrentTile >= 0)
	{
    	*pdwCurrentTile = iCurrentTile;
	}
	else
	{
        iCurrentTile = *pdwCurrentTile = 0;
		*pdwStart       = 0;
	}

    dwTempScreenRes  = dwScreenResolution;
    dwTempScreenRes -= (dwTileSize - *pdwStart);
	dwTilesToDraw++;

	while(dwTempScreenRes)
	{
		dwTilesToDraw++;

        if(dwTempScreenRes >= dwTileSize)
		{
            dwTempScreenRes -= dwTileSize;
		}
		else
		{
			dwTempScreenRes = 0;
		}
	}
    
	if(dwTileResolution <  (*pdwCurrentTile + dwTilesToDraw))
	{
		dwTempScreenRes = dwScreenResolution;
     	dwTilesToDraw   = 0;
		*pdwStart       = 0;

		while(dwTempScreenRes)
		{
			dwTilesToDraw++;

			if(dwTempScreenRes >= dwTileSize)
			{
				dwTempScreenRes -= dwTileSize;
			}
			else
			{
                *pdwStart = dwTileSize - dwTempScreenRes;
                dwTempScreenRes = 0;
            }
		}

		*pdwCurrentTile = dwTileResolution - dwTilesToDraw;

	}

    return dwTilesToDraw;
}


/***********************************************************************
 * Tile_CreateTileSprite
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
HTILESPRITE Tile_CreateTileSprite(HTILE hTile, PTILE_SPRITE pTileSprite)
{
	PTILEENG pTileEng = (PTILEENG)hTile;
	PTILE_SPRITE_ENG pTileSpriteEng = NULL;

	if(pTileSpriteEng = (PTILE_SPRITE_ENG)LocalAlloc(LMEM_ZEROINIT, sizeof(TILE_SPRITE_ENG)))
	{
		pTileSpriteEng->pNextSprite = pTileEng->pSprites;
		pTileEng->pSprites = pTileSpriteEng;

		pTileSprite->dwSpriteId    = pTileSprite->dwSpriteId;
		pTileSprite->pTileContext  = pTileSprite->pTileContext;
		pTileSprite->pfnDrawSprite = pTileSprite->pfnDrawSprite;
		
		pTileSprite->pfnSpriteCollision = pTileSprite->pfnSpriteCollision;
		
		pTileSprite->dwCurrentTileX = pTileSprite->dwCurrentTileX;
		pTileSprite->dwCurrentTileY = pTileSprite->dwCurrentTileY;

		pTileSprite->dwViewWidth  = pTileSprite->dwViewWidth;
		pTileSprite->dwViewHeight = pTileSprite->dwViewHeight;
		pTileSprite->dwFlags      = pTileSprite->dwFlags;
		pTileSprite->pTileEng     = pTileSprite->pTileEng;
	}

    return (HTILESPRITE)pTileSpriteEng;
}


/***********************************************************************
 * TileSprite_MoveUp
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
BOOL WINAPI TileSprite_MoveUp(HTILESPRITE hTileSprite, DWORD dwPosition)
{
	PTILE_SPRITE_ENG pTileSpriteEng = (PTILE_SPRITE_ENG)hTileSprite;
}


/***********************************************************************
 * TileSprite_MoveDown
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
BOOL WINAPI TileSprite_MoveDown(HTILESPRITE hTileSprite, DWORD dwPosition)
{
	PTILE_SPRITE_ENG pTileSpriteEng = (PTILE_SPRITE_ENG)hTileSprite;
}

/***********************************************************************
 * TileSprite_MoveLeft
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
BOOL WINAPI TileSprite_MoveLeft(HTILESPRITE hTileSprite, DWORD dwPosition)
{
	PTILE_SPRITE_ENG pTileSpriteEng = (PTILE_SPRITE_ENG)hTileSprite;
}


/***********************************************************************
 * TileSprite_MoveRight
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
BOOL WINAPI TileSprite_MoveRight(HTILESPRITE hTileSprite, DWORD dwPosition)
{
	PTILE_SPRITE_ENG pTileSpriteEng = (PTILE_SPRITE_ENG)hTileSprite;
}


/***********************************************************************
 * Tile_DestroyTileSprite
 *  
 *    
 *    Find the current start position of what tiles to start drawing.
 *    
 *
 *
 ***********************************************************************/
void Tile_DestroyTileSprite(HTILESPRITE hTileSprite)
{
	PTILE_SPRITE_ENG pTileSpriteEng = (PTILE_SPRITE_ENG)hTileSprite;

	if(pTileSpriteEng->pNextSprite)
	{
		pTileSpriteEng->pNextSprite->pPrevSprite = pTileSpriteEng->pPrevSprite;
	}

	if(pTileSpriteEng->pPrevSprite)
	{
		pTileSpriteEng->pPrevSprite->pNextSprite = pTileSpriteEng->pNextSprite;
	}

	if(pTileSpriteEng->pTileEng->pSprites == pTileSpriteEng)
	{
		pTileSpriteEng->pTileEng->pSprites = pTileSpriteEng->pNextSprite;
	}

	LocalFree(pTileSpriteEng);
}


/***********************************************************************
 * Tile_Debug
 *  
 *    Debug 
 *
 *    
 *
 * Parameters
 *     Debug
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
 void Tile_Debug(char *pszFormatString, ...)
 {
     char DebugString[256];
     va_list vl;

     va_start(vl, pszFormatString);
     vsprintf(DebugString, pszFormatString, vl);
     va_end(vl);

     OutputDebugStringA(DebugString);
 }
